clear all; clc;

%% SIMULATION PARAMETERS
T = 30;                 %Number of years of the projection (we need T>20!)
iter = 10000;           %Number of interations to simulate the projection (should be the same as number of simulated paths of macroeconomic variables)
deter = 0;%1;           %Indicator of deterministic projections (1) or stochastic (0)
zlb = 0;%1;             %Indicator whether the zlb is accounted for (not in the paths of macro vars but in the subsequent simulation only are imposed zero for negtive values of the rates)
shareEUR = 0.7;         %Share of EUR assets in portfolio, the rest comprises of USD assets
shareBonds = 0.9;       %Share of 1-y bonds in portfolio, the rest comprises of stocks

%% INPUT DATA

%BENCHMARK: macro data as of Inflation Forecast 2018Q1, balance sheet data are as of 31.12.2017, yields as of 29.12.2017
first  = '2018q1';          %first quarter of simulation, i.e. first projectionpoint is for year 2018, i.e. 2018Q1-2018Q4
obs_y = 0;                  %how many years ahead of macro outlook are already observed data?
obs_for = 0;
nGDP0  = 5058883;           %nominal GDP (mil CZK) sum over last 4 quarters (2017Q1-2017Q4)

%Balance sheet items:
FX0   = 3156578.3 - 171125.9;       %FX assets - FX liabilities, milCZK 
Cash0 =  591726;                    %Currency in circulation, milCZK 
Dep0  = 2566182.9 + 17536.8...      %Deposits of residents (Total) + Other liabilities 
      - 4703.6 - 2494.7 - 10724.9;  % - Other assets - Loans to residents - Fixed assets, milCZK 
OC0   = 0.00045*nGDP0;              %Operational costs, milCZK, computed as a fraction of nominal GDP
Eq0   = -172070.1;                  %Equity

%outlook of interest rates, return on stocks and exchange rates
i_CZ = [0.74129,1.51638];           %2W repo: quarterly average of daily average in forthcoming 4 quarters (i.e. 2018Q1-2018Q4 and 2019Q1-2019Q4)
    
%outlook of returns on EUR bonds - return on 1Y EUR bonds: g3 outlook of euribor 3M (6years) + average (over 2004q-2007q4, 2009q1-2017q4) spread betwen euribor 3M and 1Y government AAA bond yields  
i_EUR = [-0.476+0.3,-0.05+0.3,0.67+0.3,1.73+0.3,2.70+0.3,3.17+0.3,3.50+0.3,3.50+0.3,3.50+0.3,3.50+0.3];   
    
%outlook of returns on USD bonds - ACM - see ACMterm_premia.xls
yield_curveUSD = [1.737667859,1.87155836,1.998846785,2.101872867,2.183700192,2.250292048,2.306226127,2.354323227,2.396126344,2.432390095];
term_premiaUSD = [-0.364117837,-0.469465213,-0.491200261,-0.496998961,-0.503316529,-0.512348633,-0.523272686,-0.535222056,-0.547821013,-0.561108941];
i_USD = bond_yields(yield_curveUSD,term_premiaUSD);
    
e_EUR = [-3.75731,-0.5414];         %CZK/EUR yoy - first two years
e_USD = [-2.74096,-0.88632];        %CZK/USD yoy
nGDP_gr = [6.47589,5.28363];        %nominal GDP growth

%EQUILIBRIUM VALUES (EQ - equilibrium values after 2 years, LR means long run
%value after the end of convergence or entering the eurozone
i_CZ_EQ     = 3 - 0.24;         % 2T repo = eq 3M pribor assumed in g3 - premium (average over observed values)
i_CZ_LR     = 3.5 - 0.24;       % 2T repo
i_EUR_EQ    = 3.5 + 0.3;        % 1Y bond interest rate = 3M EUROBOR - averate spread 3M and 1Y bond
i_USD_EQ    = 3.5 + 0.3;        % 1Y bond interest rate
e_EUR_EQ    = -1.5;
e_USD_EQ    = -1.5;
e_EUR_LR    = 0;
e_USD_LR    = 0;
nGDP_gr_EQ  = 5;
nGDP_gr_LR  = 3;
stocks = 6;                     %return on stocks (%)

%EXOGENOUS CHANGE IN ASSETS (sales of FX assets, normalization of balance sheet etc.)
% it is a vector of length T
% you can either switch to sales of FX income after FXsales_years years
FXsales = 0;%0;1;
FXsales_years = 2;%  % FXsales_years = 2 sales start in 2020
% or you can directly impose the amount of FX change (minus sign means
% sales) - the same for all i's
% NOTE that in this case FXsale has to be equal 0;
FXInt = zeros(iter,T); 

%CHANGE IN MONEY IN CIRCULATION - input a ration between nominal GDP growth and growth of currency in circulation
monetization = ones(1,T);       %NO CHANGE (benchmark)
%monetization = linspace(1,0.5,30);
%monetization = linspace(1,-2,30); %Sweden

%% Preliminaries

%load simulated paths if stochastic setting is requested
% f_paths includes: [(1) foreign demand, (2) foreign inflation, (3) 3m euribor, (4) real GDP, (5) MPR inflation, (6) pribor 3m, (7) CZK/EUR growth
%                    (8) SXXP stock return, (9) SNP stock return, (10) CZK/USD] 
nvars = 10; %number of vars from the macro models
if deter
    f_paths = cell(nvars,1);
    for i=1:nvars
        f_paths{i,1}=zeros(iter,T);
    end
else
    load('bvar\sim_paths1_symm.mat');   %BENCHMARK  
        
    if obs_y>0
        for i=[4,5,6,7,10]
            f_paths{i,1}(1:iter,1:obs_y)=zeros(iter,obs_y);
        end
    end
    if obs_for>0
        for i=[3]
            f_paths{i,1}(1:iter,1:obs_for)=zeros(iter,obs_for);
        end
    end
end

%diagnostics of forecast paths
%{
disp('F_paths: Skewness');
tmp=[];
for i=1:size(f_paths,1)
   tmp = [tmp max(abs(skewness(f_paths{i,1})))]; 
end
tmp

disp('F_paths: median');
tmp=[];
for i=1:size(f_paths,1)
   tmp = [tmp max(abs(median(f_paths{i,1})))]; 
end
tmp
%}

load('dates');
dates = dates(find(strcmp(first,dates))+4:4:find(strcmp(first,dates))+123,1);

FX_paths        = NaN(iter,T+1);
FX2GDP_paths    = NaN(iter,T+1);
Inc_paths       = NaN(iter,T);
Exp_paths       = NaN(iter,T);
Dep_paths       = NaN(iter,T+1);
OC_paths        = NaN(iter,T+1);
Cash_paths      = NaN(iter,T+1);
Liab_paths      = NaN(iter,T+1);
Prft_paths      = NaN(iter,T);
Prft2GDP_paths  = NaN(iter,T);
Eq_paths        = NaN(iter,T+1);
Eq2GDP_paths    = NaN(iter,T+1);
CD_ratio        = NaN(iter,T+1);
yieldEUR        = NaN(iter,T);
yieldUSD        = NaN(iter,T);
nGDP            = NaN(iter,T+1);

%% SET DETERMINISTIC OUTLOOK ('MODAL' PATH) OF ALL EXO VARS FOR T PERIODS AHEAD

%CZ interest rate
i_CZ = [i_CZ,linspace(i_CZ_EQ,i_CZ_LR,18-size(i_CZ,2)+2),i_CZ_LR,i_CZ_LR*ones(1,T-21)];

%foreign interest rates
i_EUR = [i_EUR(1,1:end-1),linspace(i_EUR(1,end),i_EUR_EQ,11),i_EUR_EQ*ones(1,T-20)];
i_USD = [i_USD(1,1:end-1),linspace(i_USD(1,end),i_USD_EQ,11),i_USD_EQ*ones(1,T-20)];

%exchange rates
e_EUR = [e_EUR,linspace(e_EUR_EQ,e_EUR_LR,18-size(e_EUR,2)+2),e_EUR_LR,e_EUR_LR*ones(1,T-21)];
e_USD = [e_USD,linspace(e_USD_EQ,e_USD_LR,18-size(e_USD,2)+2),e_USD_LR,e_USD_LR*ones(1,T-21)];

%nominal GDP growth
nGDP_gr = [nGDP_gr,linspace(nGDP_gr_EQ,nGDP_gr_LR,18-size(nGDP_gr,2)+2),nGDP_gr_LR,nGDP_gr_LR*ones(1,T-21)];

%stock returns
stock_ret = stocks*ones(1,30);


%% SIMULATION
for i=1:iter
    i_CZ_paths(i,:)     = i_CZ  + f_paths{6,1}(i,:);
    i_EUR_paths(i,:)    = i_EUR + f_paths{3,1}(i,:);
    
    %i_USD_paths(i,:)    = i_USD + f_paths{3,1}(iter-i+1,:);           %use different (but random) path for euro to define USD path
    i_USD_paths(i,:)    = i_USD + f_paths{3,1}(i,:);                    %use the same path for euro to define USD path
    %st_retUSD_paths(i,:)= stock_ret + f_paths{9,1}(iter-i+1,:);
    st_retUSD_paths(i,:)= stock_ret + f_paths{9,1}(i,:);
    
    e_EUR_paths(i,:)    = e_EUR + f_paths{7,1}(i,:);
    e_USD_paths(i,:)    = e_USD + f_paths{10,1}(i,:);                  
    nGDP_gr_paths(i,:)  = nGDP_gr + f_paths{4,1}(i,:) + f_paths{5,1}(i,:); 
    st_retEUR_paths(i,:)= stock_ret + f_paths{8,1}(i,:);
    
    
    if zlb
        i_CZ_paths(i,:) = max(i_CZ_paths(i,:),0);
        i_EUR_paths(i,:) = max(i_EUR_paths(i,:),0);
        i_USD_paths(i,:) = max(i_USD_paths(i,:),0);
    end
    
    FX_paths(i,1)    = FX0;
    FX2GDP_paths(i,1)= FX0/nGDP0;
    Dep_paths(i,1)   = Dep0;
    OC_paths(i,1)    = OC0;
    Cash_paths(i,1)  = Cash0;
    Liab_paths(i,1)  = Cash0 + Dep0;
    Eq_paths(i,1)    = Eq0; 
    Eq2GDP_paths(i,1)= Eq0/nGDP0;
    CD_ratio(i,1)    = Cash0/Dep0;
    nGDP(i,1) = nGDP0;
    
    yieldEUR(i,:) = shareBonds*i_EUR_paths(i,:) + (1-shareBonds)*st_retEUR_paths(i,:);       %outlook of EUR portfolio yields
    yieldUSD(i,:) = shareBonds*i_USD_paths(i,:) + (1-shareBonds)*st_retUSD_paths(i,:);       %outlook of USD portfolio yields
        
    for t=1:T
        %Preliminary
        nGDP(i,t+1) = (1+0.01*nGDP_gr_paths(i,t))*nGDP(i,t); 
        OC_paths(i,t+1) = (1+0.01*nGDP_gr_paths(i,t))*OC_paths(i,t);    %Operation costs
        
        %Income statement - Income and Expenses
        Inc_paths(i,t)  = shareEUR*0.01*(yieldEUR(i,t) + e_EUR_paths(i,t))*FX_paths(i,t) + ...
                          (1-shareEUR)*0.01*(yieldUSD(i,t) + e_USD_paths(i,t))*FX_paths(i,t);
        
        if FXsales && t>FXsales_years
           FXInt(i,t) = -Inc_paths(i,t);
        end
                      
        Exp_paths(i,t)  = 0.01*i_CZ_paths(i,t)*Dep_paths(i,t) + OC_paths(i,t+1);
        
        %Balance sheet items - assets    
        FX_paths(i,t+1) = shareEUR*(1 + 0.01*yieldEUR(i,t) + 0.01*e_EUR_paths(i,t))*FX_paths(i,t) + ...
                          (1-shareEUR)*(1 + 0.01*yieldUSD(i,t) + 0.01*e_USD_paths(i,t))*FX_paths(i,t) + FXInt(i,t);
                      
        %Balance sheet items - liabilities     
        Liab_paths(i,t+1) = Liab_paths(i,t) + Exp_paths(i,t) + FXInt(i,t);
        Cash_paths(i,t+1) = (1+0.01*monetization(1,t)*nGDP_gr_paths(i,t))*Cash_paths(i,t);
        Dep_paths(i,t+1)  = Liab_paths(i,t+1) - Cash_paths(i,t+1);
                           
        %Profit/loss and equity
        Prft_paths(i,t)     = Inc_paths(i,t) - Exp_paths(i,t);
        Eq_paths(i,t+1)     = Eq_paths(i,t) + Prft_paths(i,t);

        %other indicators
        FX2GDP_paths(i,t+1)  = FX_paths(i,t+1)/nGDP(1,t+1);
        CD_ratio(i,t+1)      = Cash_paths(i,t+1)/Dep_paths(i,t+1);
        Prft2GDP_paths(i,t)  = Prft_paths(i,t)/nGDP(1,t+1);
        Eq2GDP_paths(i,t+1)  = Eq_paths(i,t+1)/nGDP(1,t+1); 
        
        %check Eq = FX - Liab: should be always zero: serve also as a check
        %if initial values for balance sheet iterms are coherent
        %Check(i) = Eq_paths(i,t) - FX_paths(i,t) + Liab_paths(i,t)
    end
end
 

%% OUTPUT AND FIGURES

disp('Prob of balance size/GDP over 100%');
(sum(FX2GDP_paths>1,1)/iter)'

disp('Prob of loss/GDP over 5%');
(sum(Prft2GDP_paths<-0.05,1)/iter)'

disp('Prob of negative equity');
(sum(Eq2GDP_paths<-0,1)/iter)'

disp('Expected profit:');
ExpPrft = zeros(1,30);
for t=1:30
    ExpPrft(t)=0;
    for i=1:iter
        ExpPrft(t) = ExpPrft(t) + (1/iter)*Prft_paths(i,t);
    end
end
ExpPrft'

disp('Variance of the profit');
var(Prft_paths)'

disp('Expected loss');
ExpLoss = zeros(1,30);
for t=1:30
    ExpLoss(t)=0;
    for i=1:iter
        ExpLoss(t) = ExpLoss(t) + (1/iter)*min(Prft_paths(i,t),0);
    end
end
ExpLoss'


disp('Average FX sale (bn. CZK):');
mean(FXInt)/1000
%100*mean(FXInt)./mean(nGDP(:,2:31))
for i=1:size(FXInt,1);
    FXInt2GDP(i,:) = FXInt(i,:)./nGDP(i,2:31);
end
100*mean(FXInt2GDP)

%CONFIDENCE INTERVAL PROJECTIONS - one projection in a figure
%{
clear ivar
ivar = Eq2GDP_paths(:,2:31);%FX2GDP_paths(:,2:31);%Eq2GDP_paths;%Inc_paths;%yieldEUR;%FX_paths;%Eq2GDP_paths;%Prft2GDP_paths;%CD_ratio;%FX2GDP_paths;%FX_paths;%FX2GDP_paths;Prft_paths;%Eq_paths;%Dep_paths;%Eq_paths;%FX_paths;%Inc_paths;
tit =  'Equity/GDP';%'Assets/GDP';%'Equity/GDP';%'Income';%'EUR yield';%'Assets';%'Equity/GDP';%'Profits/GDP';%'Cash/deposit';%'Assets/GDP';%'Assets';%'Assets/GDP';%'CNB profit/loss';

ivar_q = quantile(ivar,[.023, 0.16, .5, 0.84, .977]);

figure
set(gcf,'color','w');
xlims = size(ivar,2);
set(0,'DefaultAxesColorOrder',[0 0 0],...
        'DefaultAxesLineStyleOrder','-.|--|-|--|-.');
clear matm
matm(:,1) = ivar_q(1,:);
for ii=2:5
    matm(:,ii) = ivar_q(ii,:) - ivar_q(ii-1,:);
end
hold on
hh=area(matm);
r=1;
b=0;
set(hh,'LineStyle','none')
set(hh(1),'FaceColor',[1 1 1]) % white
set(hh(2),'FaceColor',[r .99 b])
set(hh(3),'FaceColor',[r .9 b])
set(hh(4),'FaceColor',[r .9 b])
set(hh(5),'FaceColor',[r .99 b])
h1 = plot(ivar_q(3,:),'-','LineWidth',1.5);
xlim([1 xlims]);
hd=gca;
set(hd,'box','on','ticklength',[0.02 0.05])
set(hd, 'Layer','top')
plot([0 xlims],[0 0],'k-');
if strcmp(tit,'Cash/deposit')
   plot([0 xlims],[CD_threshold CD_threshold],'k--'); 
end
set(gca,'XTick',1:8:xlims);
set(gca,'XTickLabel',dates(1:8:xlims,1));
l = legend([h1 hh(2) hh(3)],'median','95%','68%');
%l.FontSize=10;
title(tit);
%export_fig fx2gdp.png
%export_fig cd_ratio.png
%export_fig profits2gdp.png
%export_fig eq2gdp.png

%}

%CONFIDENCE INTERVAL PROJECTIONS - 4 variables in a picture
%{
clear ivar;
ivar(:,:,1) = Prft_paths; ivar(:,:,2) = Eq_paths(:,2:end);
ivar(:,:,3) = Inc_paths; ivar(:,:,4) = Exp_paths;
tit = {'Profit/loss','Equity','Income','Expenses'};

ivar = ivar/1000; %if the graphs to be in billions of CZK
ivar_q = quantile(ivar,[.023, 0.16, .5, 0.84, .977]);
figure
set(gcf,'color','w');
xlims = size(ivar,2);
for i=1:4
    set(0,'DefaultAxesColorOrder',[0 0 0],...
        'DefaultAxesLineStyleOrder','-.|--|-|--|-.');
    clear matm
    matm(:,1) = ivar_q(1,:,i);
    for ii=2:5
        matm(:,ii) = ivar_q(ii,:,i) - ivar_q(ii-1,:,i);
    end
    subplot(2,2,i);
    hold on
    hh=area(matm);
    r=1;
    b=0;
    set(hh,'LineStyle','none')
    set(hh(1),'FaceColor',[1 1 1]) % white
    set(hh(2),'FaceColor',[r .99 b])
    set(hh(3),'FaceColor',[r .9 b])
    set(hh(4),'FaceColor',[r .9 b])
    set(hh(5),'FaceColor',[r .99 b])
    h1 = plot(ivar_q(3,:,i),'-','LineWidth',1.5);
    xlim([1 xlims]);
    hd=gca;
    set(hd,'box','on','ticklength',[0.02 0.05])
    set(hd, 'Layer','top')
    hd.YAxis.Exponent = 0;
    ylabel('bn. CZK');
    plot([0 xlims],[0 0],'k-','LineWidth',1);
    set(gca,'XTick',1:8:xlims);
    set(gca,'XTickLabel',dates(1:8:xlims,1));
    %picked draw
    hp = plot(ivar(i_pick,:,i),'--','LineWidth',0.5);
    if i==1
        l = legend([h1 hh(2) hh(3) hp],'median','95%','68%','closest-to-median path','Orientation','horizontal');
        %l.FontSize = 12;
    end
    title(tit{i});   
end
%export_fig projections.png
%export_fig projections_draw.png
%}

%INPUT MACRO DATA
%{
clear ivar
ivar(:,:,1) = i_CZ_paths;ivar(:,:,2) = i_EUR_paths;ivar(:,:,3) = i_USD_paths;
ivar(:,:,4) = e_EUR_paths;ivar(:,:,5) = e_USD_paths;ivar(:,:,6) = nGDP_gr_paths;
ivar(:,:,7) = st_retEUR_paths;ivar(:,:,8) = st_retUSD_paths;
ivar_deter = [i_CZ;i_EUR;i_USD;e_EUR;e_USD;nGDP_gr;stock_ret;stock_ret];
tit = {'Domestic interest rate','EUR interest rate','USD interest rate','CZK/EUR change',...
       'CZK/USD change','GDP growth','Stock returns (SXXP)','Stock returns (S&P 500)'};
xlims = size(ivar,2);
figure
set(gcf,'color','w');
for i=1:8
    ivar_q = quantile(ivar(:,:,i),[.023, 0.16, .5, 0.84, .977]);
    ivar_q(3,:) =  mean(ivar(:,:,i));
    set(0,'DefaultAxesColorOrder',[0 0 0],...
        'DefaultAxesLineStyleOrder','-.|--|-|--|-.');
    clear matm
    matm(:,1) = ivar_q(1,:);
    for ii=2:5
        matm(:,ii) = ivar_q(ii,:) - ivar_q(ii-1,:);
    end
    subplot(4,2,i);
    hold on
    hh=area(matm);
    r=1;
    b=0;
    set(hh,'LineStyle','none')
    set(hh(1),'FaceColor',[1 1 1]) % white
    set(hh(2),'FaceColor',[r .99 b])
    set(hh(3),'FaceColor',[r .9 b])
    set(hh(4),'FaceColor',[r .9 b])
    set(hh(5),'FaceColor',[r .99 b])
    h1 = plot(ivar_q(3,:),'-','LineWidth',1.5);
    %h2 = plot(ivar_deter(i,:),'--','LineWidth',0.5);
    plot([0 xlims],[0 0],'k-');
    xlim([1 xlims]);
    hd=gca;
    set(hd,'box','on','ticklength',[0.02 0.05])
    set(hd, 'Layer','top')
    set(gca,'XTick',1:8:xlims);
    set(gca,'XTickLabel',dates(1:8:xlims,1));
    if i==1
        %legend([h1 hh(2) hh(3) h2],'median','95%','68%','deterministic path','Orientation','horizontal');
        l = legend([h1 hh(2) hh(3)],'median','95%','68%','Orientation','horizontal');
        l.FontSize = 10;
    end
    title(tit{i});
end
%export_fig input_series.png
%}

%MEAN/MEDIAN PROJECTIONS - 4 variables
%{
clear ivar ivar_q;
ivar(:,:,1) = Prft_paths; ivar(:,:,2) = Eq_paths(:,2:end);
ivar(:,:,3) = Inc_paths; ivar(:,:,4) = Exp_paths;
tit = {'Profit/loss','Equity','Income','Expenses'};

ivar = ivar/1000; %if the graphs to be in billions of CZK
figure
set(gcf,'color','w');
for i=1:4    
    ivar_q = quantile(ivar(:,:,i),.5);
    %ivar_q = squeeze(mean(ivar(:,:,i)));
    %ivar_q = median (ivar(:,:,i),1);
    xlims = size(ivar,2);
    subplot(2,2,i);
    hold on
    plot(ivar_q,'-','LineWidth',1.5);
    %picked draw
    %plot(ivar(i_pick,:,i),'--','LineWidth',0.5);
    if i~=4
        plot([0 xlims],[0 0]);
    end
    xlim([1 xlims]);
    hd=gca;
    hd.YAxis.Exponent = 0;
    ylabel('bn. CZK');
    set(gca,'XTick',1:8:xlims);
    set(gca,'XTickLabel',dates(1:8:xlims,1));
    title(tit{i});
end
%export_fig median_projections.png
%export_fig mean_projections.png
%}

%MEAN/MEDIAN PROJECTIONS - 2 variables
%{
clear ivar ivar_q;
ivar(:,:,1) = Prft_paths; ivar(:,:,2) = Eq_paths(:,2:end);
tit = {'Profit/loss','Equity'};

ivar = ivar/1000; %if the graphs to be in billions of CZK
figure
set(gcf,'color','w');
for i=1:2    
    ivar_q = quantile(ivar(:,:,i),.5);
    %ivar_q = squeeze(mean(ivar(:,:,i)));
    xlims = size(ivar,2);
    subplot(1,2,i);
    hold on
    plot(ivar_q,'-','LineWidth',1.5);
    if i~=4
        plot([0 xlims],[0 0]);
    end
    xlim([1 xlims]);
    hd=gca;
    hd.YAxis.Exponent = 0;
    ylabel('bn. CZK');
    set(gca,'XTick',1:8:xlims);
    set(gca,'XTickLabel',dates(1:8:xlims,1));
    title(tit{i});
end
%export_fig median_projections_2varsR.png
%}

%DENSITY FORECAST
%{
ivar=FX2GDP_paths;%FX_paths;%yieldEUR;e_EUR_paths;%FX_paths;%Eq_paths;
iperiod = 10; %years ahead
figure
hold on
histogram(ivar(:,iperiod),50);
%}

%COMPARE TWO DENSITIES
%{
bars=500;
ivar=FX_paths;%FX2GDP_paths;%FX_paths;%yieldEUR;e_EUR_paths;%FX_paths;%Eq_paths;
iperiod = 20; %years ahead

load('ivar2');

figure
hold on
    max_bin = max(max(ivar(:,iperiod)),max(ivar2(:,iperiod)));
    xbars = linspace(0,max_bin,bars);
    hist(squeeze(ivar(:,iperiod)),xbars);
    hold on;
    h = findobj(gca,'Type','patch');
    set(h,'FaceColor','r','EdgeColor','w','facealpha',0.75);
    hold on;
    hist(squeeze(ivar2(:,iperiod)),xbars);
    h1 = findobj(gca,'Type','patch');
    set(h1,'facealpha',0.75);
    xlim([0 max_bin]);
    set(gca,'YTickLabel',[]);
    legend('bonds','stocks');
    title('Assets');
    %title('Assets/GDP');
    xlim([0 2e7]);

set(gcf,'color','w');
%export_fig shock_vol.png

 
disp('1pct quantile:');
quantile(ivar(:,iperiod),0.01)
quantile(ivar2(:,iperiod),0.01)

%}
